/*
* This file is part of BIN2TRD.
*
* Copyright (C) 2026 zxcygnus <cygnus1 at post.cz> https://cygnus.speccy.cz
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or (at
* your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>		// strcpy
#include <stdint.h>		// integer types
#include "trdos_structure.h"

#define VERSION		"0.1"
#define	DEBUG		0

/* TRDOS head offsets in directory of disk image */
#define TR_FILENAME	0
#define TR_EXTENSION	8
#define TR_ADDRESS	9
#define TR_LENGTH	11
#define TR_SECTORS	13
#define TR_SECTOR	14
#define TR_TRACK	15

/* structure where will be assembled target TRDOS head */
// typedef struct {
// 	unsigned char byte[16];
// } Ttrdoshead;

/* structure for reading and converting filename of target file from parameter */
typedef struct {
	unsigned char byte[9];
} Ttrdosfilename;


/* global options */
int	verbose = 0;
int	force = 0;
int	colors = 0;
int	dryrun = 0;
int	errno;

/* selection of ANSI colors */
#define CRED     "\x1b[31m"
#define CGREEN   "\x1b[32m"
#define CYELLOW  "\x1b[33m"
#define CBLUE    "\x1b[34m"
#define CMAGEN   "\x1b[35m"
#define CCYAN    "\x1b[36m"
#define CBRED    "\x1b[91m"
#define CBGREEN  "\x1b[92m"
#define CBYELLOW "\x1b[93m"
#define CBWHITE  "\x1b[97m"
#define CRESET   "\x1b[0m"

/* CRC-32 (Ethernet, ZIP, etc.) polynomial in reversed bit order. */
#define	POLY	0xedb88320

typedef enum {none, black, red, green, yellow, white, brightred, brightgreen, brightyellow, brightwhite} listofcolors_t;


uint32_t crc32Lookup[256] = {
	0x00000000,0x77073096,0xEE0E612C,0x990951BA,0x076DC419,0x706AF48F,0xE963A535,0x9E6495A3,
	0x0EDB8832,0x79DCB8A4,0xE0D5E91E,0x97D2D988,0x09B64C2B,0x7EB17CBD,0xE7B82D07,0x90BF1D91,
	0x1DB71064,0x6AB020F2,0xF3B97148,0x84BE41DE,0x1ADAD47D,0x6DDDE4EB,0xF4D4B551,0x83D385C7,
	0x136C9856,0x646BA8C0,0xFD62F97A,0x8A65C9EC,0x14015C4F,0x63066CD9,0xFA0F3D63,0x8D080DF5,
	0x3B6E20C8,0x4C69105E,0xD56041E4,0xA2677172,0x3C03E4D1,0x4B04D447,0xD20D85FD,0xA50AB56B,
	0x35B5A8FA,0x42B2986C,0xDBBBC9D6,0xACBCF940,0x32D86CE3,0x45DF5C75,0xDCD60DCF,0xABD13D59,
	0x26D930AC,0x51DE003A,0xC8D75180,0xBFD06116,0x21B4F4B5,0x56B3C423,0xCFBA9599,0xB8BDA50F,
	0x2802B89E,0x5F058808,0xC60CD9B2,0xB10BE924,0x2F6F7C87,0x58684C11,0xC1611DAB,0xB6662D3D,
	0x76DC4190,0x01DB7106,0x98D220BC,0xEFD5102A,0x71B18589,0x06B6B51F,0x9FBFE4A5,0xE8B8D433,
	0x7807C9A2,0x0F00F934,0x9609A88E,0xE10E9818,0x7F6A0DBB,0x086D3D2D,0x91646C97,0xE6635C01,
	0x6B6B51F4,0x1C6C6162,0x856530D8,0xF262004E,0x6C0695ED,0x1B01A57B,0x8208F4C1,0xF50FC457,
	0x65B0D9C6,0x12B7E950,0x8BBEB8EA,0xFCB9887C,0x62DD1DDF,0x15DA2D49,0x8CD37CF3,0xFBD44C65,
	0x4DB26158,0x3AB551CE,0xA3BC0074,0xD4BB30E2,0x4ADFA541,0x3DD895D7,0xA4D1C46D,0xD3D6F4FB,
	0x4369E96A,0x346ED9FC,0xAD678846,0xDA60B8D0,0x44042D73,0x33031DE5,0xAA0A4C5F,0xDD0D7CC9,
	0x5005713C,0x270241AA,0xBE0B1010,0xC90C2086,0x5768B525,0x206F85B3,0xB966D409,0xCE61E49F,
	0x5EDEF90E,0x29D9C998,0xB0D09822,0xC7D7A8B4,0x59B33D17,0x2EB40D81,0xB7BD5C3B,0xC0BA6CAD,
	0xEDB88320,0x9ABFB3B6,0x03B6E20C,0x74B1D29A,0xEAD54739,0x9DD277AF,0x04DB2615,0x73DC1683,
	0xE3630B12,0x94643B84,0x0D6D6A3E,0x7A6A5AA8,0xE40ECF0B,0x9309FF9D,0x0A00AE27,0x7D079EB1,
	0xF00F9344,0x8708A3D2,0x1E01F268,0x6906C2FE,0xF762575D,0x806567CB,0x196C3671,0x6E6B06E7,
	0xFED41B76,0x89D32BE0,0x10DA7A5A,0x67DD4ACC,0xF9B9DF6F,0x8EBEEFF9,0x17B7BE43,0x60B08ED5,
	0xD6D6A3E8,0xA1D1937E,0x38D8C2C4,0x4FDFF252,0xD1BB67F1,0xA6BC5767,0x3FB506DD,0x48B2364B,
	0xD80D2BDA,0xAF0A1B4C,0x36034AF6,0x41047A60,0xDF60EFC3,0xA867DF55,0x316E8EEF,0x4669BE79,
	0xCB61B38C,0xBC66831A,0x256FD2A0,0x5268E236,0xCC0C7795,0xBB0B4703,0x220216B9,0x5505262F,
	0xC5BA3BBE,0xB2BD0B28,0x2BB45A92,0x5CB36A04,0xC2D7FFA7,0xB5D0CF31,0x2CD99E8B,0x5BDEAE1D,
	0x9B64C2B0,0xEC63F226,0x756AA39C,0x026D930A,0x9C0906A9,0xEB0E363F,0x72076785,0x05005713,
	0x95BF4A82,0xE2B87A14,0x7BB12BAE,0x0CB61B38,0x92D28E9B,0xE5D5BE0D,0x7CDCEFB7,0x0BDBDF21,
	0x86D3D2D4,0xF1D4E242,0x68DDB3F8,0x1FDA836E,0x81BE16CD,0xF6B9265B,0x6FB077E1,0x18B74777,
	0x88085AE6,0xFF0F6A70,0x66063BCA,0x11010B5C,0x8F659EFF,0xF862AE69,0x616BFFD3,0x166CCF45,
	0xA00AE278,0xD70DD2EE,0x4E048354,0x3903B3C2,0xA7672661,0xD06016F7,0x4969474D,0x3E6E77DB,
	0xAED16A4A,0xD9D65ADC,0x40DF0B66,0x37D83BF0,0xA9BCAE53,0xDEBB9EC5,0x47B2CF7F,0x30B5FFE9,
	0xBDBDF21C,0xCABAC28A,0x53B39330,0x24B4A3A6,0xBAD03605,0xCDD70693,0x54DE5729,0x23D967BF,
	0xB3667A2E,0xC4614AB8,0x5D681B02,0x2A6F2B94,0xB40BBE37,0xC30C8EA1,0x5A05DF1B,0x2D02EF8D
	};


void printzxchars (unsigned char c)
{
	/* This function converts ZX Spectrum characters for Linux console as UTF-8 */
	switch (c)
	{
		/* symbol for unknown UTF-8 character should be */
		/* U+FFFD question mark in black square		*/
		/* U+25A1 white square 				*/
		case 0 ... 31:
			printf ("□");		/* control codes will be printed as UTF-8 unknown character white square */
			break;
		case 32 ... 95:
			printf ("%c", c);	/* print directly as is */
			break;
		case 96:
			printf ("£");		/* british pound */
			break;
		case 97 ... 126:
			printf ("%c", c);	/* print directly as is */
			break;
		case 127:
			printf ("©");		/* copyright */
			break;
		case 128:
			printf (" ");		/* GRAPH - empty box*/
			break;
		case 129:
			printf ("▝");		/* GRAPH - quadrant upper right */
			break;
		case 130:
			printf ("▘");		/* GRAPH - quadrant upper left */
			break;
		case 131:
			printf ("▀");		/* GRAPH - upper half block */
			break;
		case 132:
			printf ("▗");		/* GRAPH - quadrant lower right */
			break;
		case 133:
			printf ("▐");		/* GRAPH - light half block */
			break;
		case 134:
			printf ("▚");		/* GRAPH - quadrant upper left and lower righr */
			break;
		case 135:
			printf ("▜");		/* GRAPH - quadrant upper left and upper right and lower right */
			break;
		case 136:
			printf ("▖");		/* GRAPH - quadrant lower left */
			break;
		case 137:
			printf ("▞");		/* GRAPH - quadrant upper right and lower left */
			break;
		case 138:
			printf ("▌");		/* GRAPH - left half block */
			break;
		case 139:
			printf ("▛");		/* GRAPH - quadrant upper left and upper right and lower left */
			break;
		case 140:
			printf ("▄");		/* GRAPH - lower half block */
			break;
		case 141:
			printf ("▟");		/* GRAPH - quadrant upper right and lower left and lower right */
			break;
		case 142:
			printf ("▙");		/* GRAPH - quadrant upper left and lower left and lower right */
			break;
		case 143:
			printf ("█");		/* GRAPH - full block */
			break;
		case 144 ... 164:
			printf ("%c", c-79);	/* GRAPH - same as ASCII characters 'A'...'U' */
			break;
		case 165:
			printf ("RND");		/* TOKEN - BASIC command */
			break;
		case 166:
			printf ("INKEY$");	/* TOKEN - BASIC command */
			break;
		case 167:
			printf ("PI");		/* TOKEN - BASIC command */
			break;
		case 168:
			printf ("FN ");		/* TOKEN - BASIC command */
			break;
		case 169:
			printf ("POINT ");	/* TOKEN - BASIC command */
			break;
		case 170:
			printf ("SCREEN$ ");	/* TOKEN - BASIC command */
			break;
		case 171:
			printf ("ATTR ");	/* TOKEN - BASIC command */
			break;
		case 172:
			printf ("AT ");		/* TOKEN - BASIC command */
			break;
		case 173:
			printf ("TAB ");	/* TOKEN - BASIC command */
			break;
		case 174:
			printf ("VAL$ ");	/* TOKEN - BASIC command */
			break;
		case 175:
			printf ("CODE ");	/* TOKEN - BASIC command */
			break;
		case 176:
			printf ("VAL ");	/* TOKEN - BASIC command */
			break;
		case 177:
			printf ("LEN ");	/* TOKEN - BASIC command */
			break;
		case 178:
			printf ("SIN ");	/* TOKEN - BASIC command */
			break;
		case 179:
			printf ("COS ");	/* TOKEN - BASIC command */
			break;
		case 180:
			printf ("TAN ");	/* TOKEN - BASIC command */
			break;
		case 181:
			printf ("ASN ");	/* TOKEN - BASIC command */
			break;
		case 182:
			printf ("ACS ");	/* TOKEN - BASIC command */
			break;
		case 183:
			printf ("ATN ");	/* TOKEN - BASIC command */
			break;
		case 184:
			printf ("LN ");		/* TOKEN - BASIC command */
			break;
		case 185:
			printf ("EXP ");	/* TOKEN - BASIC command */
			break;
		case 186:
			printf ("INT ");	/* TOKEN - BASIC command */
			break;
		case 187:
			printf ("SQR ");	/* TOKEN - BASIC command */
			break;
		case 188:
			printf ("SGN ");	/* TOKEN - BASIC command */
			break;
		case 189:
			printf ("ABS ");	/* TOKEN - BASIC command */
			break;
		case 190:
			printf ("PEEK ");	/* TOKEN - BASIC command */
			break;
		case 191:
			printf ("IN ");		/* TOKEN - BASIC command */
			break;
		case 192:
			printf ("USR ");	/* TOKEN - BASIC command */
			break;
		case 193:
			printf ("STR$ ");	/* TOKEN - BASIC command */
			break;
		case 194:
			printf ("CHR$ ");	/* TOKEN - BASIC command */
			break;
		case 195:
			printf ("NOT ");	/* TOKEN - BASIC command */
			break;
		case 196:
			printf ("BIN ");	/* TOKEN - BASIC command */
			break;
		case 197:
			printf ("OR ");		/* TOKEN - BASIC command */
			break;
		case 198:
			printf ("AND ");	/* TOKEN - BASIC command */
			break;
		case 199:
			printf ("<=");		/* TOKEN - BASIC command */
			break;
		case 200:
			printf (">=");		/* TOKEN - BASIC command */
			break;
		case 201:
			printf ("<>");		/* TOKEN - BASIC command */
			break;
		case 202:
			printf ("LINE ");	/* TOKEN - BASIC command */
			break;
		case 203:
			printf ("THEN ");	/* TOKEN - BASIC command */
			break;
		case 204:
			printf ("TO ");		/* TOKEN - BASIC command */
			break;
		case 205:
			printf ("STEP ");	/* TOKEN - BASIC command */
			break;
		case 206:
			printf ("DEF FN ");	/* TOKEN - BASIC command */
			break;
		case 207:
			printf ("CAT ");	/* TOKEN - BASIC command */
			break;
		case 208:
			printf ("FORMAT ");	/* TOKEN - BASIC command */
			break;
		case 209:
			printf ("MOVE ");	/* TOKEN - BASIC command */
			break;
		case 210:
			printf ("ERASE ");	/* TOKEN - BASIC command */
			break;
		case 211:
			printf ("OPEN #");	/* TOKEN - BASIC command */
			break;
		case 212:
			printf ("CLOSE #");	/* TOKEN - BASIC command */
			break;
		case 213:
			printf ("MERGE ");	/* TOKEN - BASIC command */
			break;
		case 214:
			printf ("VERIFY ");	/* TOKEN - BASIC command */
			break;
		case 215:
			printf ("BEEP ");	/* TOKEN - BASIC command */
			break;
		case 216:
			printf ("CIRCLE ");	/* TOKEN - BASIC command */
			break;
		case 217:
			printf ("INK ");	/* TOKEN - BASIC command */
			break;
		case 218:
			printf ("PAPER ");	/* TOKEN - BASIC command */
			break;
		case 219:
			printf ("FLASH ");	/* TOKEN - BASIC command */
			break;
		case 220:
			printf ("BRIGHT ");	/* TOKEN - BASIC command */
			break;
		case 221:
			printf ("INVERSE ");	/* TOKEN - BASIC command */
			break;
		case 222:
			printf ("OVER ");	/* TOKEN - BASIC command */
			break;
		case 223:
			printf ("OUT ");	/* TOKEN - BASIC command */
			break;
		case 224:
			printf ("LPRINT ");	/* TOKEN - BASIC command */
			break;
		case 225:
			printf ("LLIST ");	/* TOKEN - BASIC command */
			break;
		case 226:
			printf ("STOP ");	/* TOKEN - BASIC command */
			break;
		case 227:
			printf ("READ ");	/* TOKEN - BASIC command */
			break;
		case 228:
			printf ("DATA ");	/* TOKEN - BASIC command */
			break;
		case 229:
			printf ("RESTORE ");	/* TOKEN - BASIC command */
			break;
		case 230:
			printf ("NEW ");	/* TOKEN - BASIC command */
			break;
		case 231:
			printf ("BORDER ");	/* TOKEN - BASIC command */
			break;
		case 232:
			printf ("CONTINUE ");	/* TOKEN - BASIC command */
			break;
		case 233:
			printf ("DIM ");	/* TOKEN - BASIC command */
			break;
		case 234:
			printf ("REM ");	/* TOKEN - BASIC command */
			break;
		case 235:
			printf ("FOR ");	/* TOKEN - BASIC command */
			break;
		case 236:
			printf ("GO TO ");	/* TOKEN - BASIC command */
			break;
		case 237:
			printf ("GO SUB ");	/* TOKEN - BASIC command */
			break;
		case 238:
			printf ("INPUT ");	/* TOKEN - BASIC command */
			break;
		case 239:
			printf ("LOAD ");	/* TOKEN - BASIC command */
			break;
		case 240:
			printf ("LIST ");	/* TOKEN - BASIC command */
			break;
		case 241:
			printf ("LET ");	/* TOKEN - BASIC command */
			break;
		case 242:
			printf ("PAUSE ");	/* TOKEN - BASIC command */
			break;
		case 243:
			printf ("NEXT ");	/* TOKEN - BASIC command */
			break;
		case 244:
			printf ("POKE ");	/* TOKEN - BASIC command */
			break;
		case 245:
			printf ("PRINT ");	/* TOKEN - BASIC command */
			break;
		case 246:
			printf ("PLOT ");	/* TOKEN - BASIC command */
			break;
		case 247:
			printf ("RUN ");	/* TOKEN - BASIC command */
			break;
		case 248:
			printf ("SAVE ");	/* TOKEN - BASIC command */
			break;
		case 249:
			printf ("RANDOMIZE ");	/* TOKEN - BASIC command */
			break;
		case 250:
			printf ("IF ");		/* TOKEN - BASIC command */
			break;
		case 251:
			printf ("CLS ");	/* TOKEN - BASIC command */
			break;
		case 252:
			printf ("DRAW ");	/* TOKEN - BASIC command */
			break;
		case 253:
			printf ("CLEAR ");	/* TOKEN - BASIC command */
			break;
		case 254:
			printf ("RETURN ");	/* TOKEN - BASIC command */
			break;
		case 255:
			printf ("COPY ");	/* TOKEN - BASIC command */
			break;
	}
}

/* separated in function, because this will be different for different operating systems */
void colorize (listofcolors_t c)
{
	/* only if colors are globally enabled */
	if (colors)
	{
		/* TODO: different colors for terminals with white background */
		switch (c)
		{
			case red:
				printf ("%s", CRED);
				break;
			case green:
				printf ("%s", CGREEN);
				break;
			case brightgreen:
				printf ("%s", CBGREEN);
				break;
			case brightyellow:
				printf ("%s", CBYELLOW);
				break;
			case brightwhite:
				printf ("%s", CBWHITE);
				break;
			default:
				printf ("%s", CRESET);
				break;
		}
	}
}

void colored_warning (void)
{
	colorize (brightyellow);
	printf ("Warning: ");
	colorize (none);
}

void colored_error (void)
{
	colorize (red);
	printf ("Error! ");
	colorize (none);
}

int bintotrd (char* source_filename, char* target_filename, Ttrdosfilename trdos_filename, uint16_t start_address)
{
	int i;			/* temporary index */
	int trdosfile;		/* counter for processed filename in TRDOS directory */
	
	int binsize;		/* length of binary file */
	int trdimagesize;	/* length of TRDOS image file */
	FILE *sourcefile;
	FILE *targetfile;
	uint8_t byte;		/* will be used for copying of binary data from source file in to target TRDOS image */

	trdos_sector sector;
	trdos_track track;

	uint8_t s;		/* variable for searching filenames - sector number */
	uint8_t o;		/* variable for searching filenames - offset in sector */
	int dupfound;		/* variable for searching filenames - will be 1 if duplicate file found in TRDOS image */
	int different;		/* variable for searching filenames - will be 1 if something in compared file differs */

	uint8_t sector_number;	/* used during writin file on TRD disk image */
	uint8_t track_number;
	
	uint16_t free_sectors;		/* in standard TRDOS disks up to 2455 */
	uint16_t file_length;		/* length of binary file, also byte counter when writing in TRDOS disk image */
	uint8_t trdos_files;		/* 0..128 */
	uint8_t trdos_deleted_files;	/* 0..127 - at least one file must not be deleted if deleted files  are preserved */
	uint8_t sector_length;		/* counter for length in sectors */
	
	unsigned int directory_offset;	/* used when writing header in TRDOS directory */
	unsigned int dir_sector;
	int fatal_error_found;

	if (DEBUG)
	{
		printf ("DEBUG source_filename\t\"%s\"\n", source_filename);
		printf ("DEBUG target_filename\t\"%s\"\n", target_filename);
		
		printf ("DEBUG trdos_filename\t\"");
		for (i = 0; i < 8; i++) printzxchars(trdos_filename.byte[i]);
		printf (".");
		printzxchars(trdos_filename.byte[8]);
		printf ("\"\n");
		
		printf ("DEBUG start_address\t%hu\n", start_address);
	}
	
	/* binary file which should be added to the TRDOS disk image */
	sourcefile = fopen (source_filename, "r");
	if (sourcefile == NULL)
	{
		colored_error ();
		printf ("Can't open source binary file %s.\n", source_filename);
		exit (EXIT_FAILURE);
	}

	/* target TRDOS disk image */
	targetfile = fopen (target_filename, "ra+");
	if (targetfile == NULL)
	{
		colored_error ();
		printf ("Can't open target disk image %s.\n", target_filename);
		exit (EXIT_FAILURE);
	}
	
	/* check length of source binary file, cannot be bigger than 256*255=65280 bytes, because maximum lenght in sectors can be 255 */
	fseek (sourcefile, 0, SEEK_END);
	binsize = ftell (sourcefile);
	fseek (sourcefile, 0, SEEK_SET);	/* seek to the start of file */
	
	if (verbose) printf ("Size of source binary file is %i bytes.\n", binsize);
	
	if (binsize > 255*256)
	{
		colored_error ();
		printf ("Source binary file is too big (%i bytes). TRDOS have limit for maximum file lenght 255 sectors (65280 bytes).\n", binsize);
		exit (EXIT_FAILURE);
	}
	
	/* check length of target file, cannot be bigger than biggest standard TRDOS format */ 
	fseek (targetfile, 0, SEEK_END);
	trdimagesize = ftell (targetfile);
	fseek (targetfile, 0, SEEK_SET);	/* seek to the start of file */
	
	if (verbose) printf ("Size of target TRDOS image file is %i bytes.\n", trdimagesize);
	
	if (trdimagesize > 80*2*16*256)
	{
		colored_error ();
		printf ("Target TRDOS image is too big (%i bytes), this program cannot work with non standard TRDOS formats. Expected biggest format is 80 tracks * 2 sides * 16 sectors = 640kB.\n", trdimagesize);
		exit (EXIT_FAILURE);
	}
	
	if (trdimagesize % 256 != 0)
	{
		colored_error ();
		printf ("Size of TRDOS image file must be aligned to 256 bytes, because it must contain whole sectors which contains 256 bytes each.\n", trdimagesize);
		exit (EXIT_FAILURE);
	}
	
	if (trdimagesize < 40*16*256)
	{
		colored_error ();
		printf ("Target TRDOS image is smaller (only %i bytes) than smallest standard format 40 tracks * 16 sectors = 160kB.\n", trdimagesize);
		exit (EXIT_FAILURE);
	}
	
	/* read TRDOS directory from target TRDOS image */
	if (fread (&track, sizeof (trdos_track), 1, targetfile) == 1)
	{
		/*  check content of TRDOS image, identifications, free space etc... */
		fatal_error_found = 0;

		/* TRDOS image must contain TRDOS identification byte in system sector */
		if (track.sector[INFO_SEC].byte[OFFSET_TRDOSIDENT] != 16)
		{
			if (!force)
			{
				colored_warning ();
				printf ("Missing TRDOS identification in target image file! You can try -f (force) parameter, but i strongly recommend to backup target file and check again if it really is TRDOS image.\n");
				fatal_error_found = 1;
			}
			else
			{
				if (verbose) printf ("Missing TRDOS identification, but ignoring it, because option --force was used.\n");
			}
		}
		
		/* TRDOS image must have specified formated capacity - only 4 values are allowed */
		if (track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT] != 22 && track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT] != 23 && track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT] != 24 && track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT] != 25)
		{
			if (!force)
			{
				colored_warning ();
				printf ("Value of formated capacity is incorrect %hhu, should be 22, 23, 24, or 25.\n", track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT]);
				fatal_error_found = 1;
			}
			else
			{
				if (verbose) printf ("Value of formated capacity is incorrect, but ignoring it, because option --force was used.\n");
			}
		}
		else
		{
			if (verbose)
			{
				printf ("Format of TRDOS image ");
				switch (track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT])
				{
					case 22:
						printf ("80 tracks double sided disk (640kB)\n");
						break;
					case 23:
						printf ("40 tracks double sided disk (320kB)\n");
						break;
					case 24:
						printf ("80 tracks single sided disk (320kB)\n");
						break;
					case 25:
						printf ("40 tracks single sided disk (160kB)\n");
						break;
				}
			}
		}
		
		/* check number of files on TRDOS image, must be in range 0..128 because 16*128=2048 bytes */
		if (track.sector[INFO_SEC].byte[OFFSET_FILES] > 128)
		{
			if (!force)
			{
				colored_warning ();
				printf ("Number of files must be in range from 0 to 128, but is %hhu.\n", track.sector[INFO_SEC].byte[OFFSET_FILES]);
				fatal_error_found = 1;
			}
			else
			{
				if (verbose) printf ("Number of files isn't in range from 0 to 128, but ignoring it, because option --force was used.\n");
			}
		}
		else
		{
			if (verbose)
			{
				printf ("\tFiles %hhu \n", track.sector[INFO_SEC].byte[OFFSET_FILES]);
				printf ("\tFree %hu sectors \n", track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] + 256*track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1]);
			}
		}
		
		
		/* check first free sector, must be in range 0..15 */
		if (track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR] > 15 || track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR] < 0)
		{
			if (!force)
			{
				colored_warning ();
				printf ("First free sector must be in range from 0 to 15, is %hhu.\n", track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR]);
				fatal_error_found = 1;
			}
			else
			{
				if (verbose) printf ("First free sector isn't in range from 0 to 15, but ignoring it, because option --force was used.\n");
			}
		}
		
		/* first free track should be up to 159... */

		/* TRDOS image must have enough of free space */
		if (track.sector[INFO_SEC].byte[OFFSET_FILES] == 128)
		{
			colored_error ();
			printf ("Directory is full, no space to write any file.\n");
			fatal_error_found = 1;
		}
		if ((track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] == 0) && (track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] == 0))
		{
			colored_error ();
			printf ("Disk is full, no space to write any file.\n");
			fatal_error_found = 1;
		}
		if (binsize > 256*(track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] + 256*track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1]))
		{
			colored_error ();
			printf ("Binary file is bigger (%hhu sectors) than remaining free space (%hhu sectors).\n", binsize/256+binsize%256, track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] + 256*track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1]);
			fatal_error_found = 1;
		}

		/* check for duplicity, print warning if file with same name already exists in disk image */
		dupfound = 0;
		for (trdosfile = 0; trdosfile < track.sector[INFO_SEC].byte[OFFSET_FILES]; trdosfile++)
		{
			/* check filename with extension, if match, compare length and start address */
			/* maybe option checksum can be used to compare content of files? */
			
			/* print filename with proper conversion of semigraphics and BASIC tokens */
			s = trdosfile / 16;		/* sector number in s - 16 bytes per item in directory */
			o = (trdosfile % 16)*16;	/* offset of filename in sector */
			
			/* print searched filenames when debugging */
			if (DEBUG) 
			{
				printf ("++ checking %hu,%hu\t[", s, o);
				for (i = 0; i < 8; i++) printzxchars(track.sector[s].byte[o+i]);
				printf (".");
				printzxchars(track.sector[s].byte[TR_EXTENSION]);
				printf ("]\n");
			}
			
			/* compare filenames include extension*/
			different = 0;
			for (i = 0; i < 9; i++)
			{
				if (trdos_filename.byte[i] != track.sector[s].byte[o+i]) different = 1;
			}
			
			/* compare length */
			if (binsize % 256 != track.sector[s].byte[o+TR_LENGTH]) different = 1;
			if (binsize / 256 != track.sector[s].byte[o+TR_LENGTH+1]) different = 1;
			
			/* if after all checks no difference was found, then it is duplicity, source file is already in TRDOS image */
			if (different == 0)
			{
				/* first duplicity found is enough to break for cycle */
				dupfound = 1;
				break;
			}
		}
		
		/* warning if duplicate file found in directory. */
		if (dupfound == 1)
		{
			if (!force)
			{
				colored_warning ();
				printf ("File with same name \"");
				
				/* print which name on TRDOS image it is */
				for (i = 0; i < 8; i++) printzxchars(track.sector[s].byte[o+i]);
				printf (".");
				printzxchars(track.sector[s].byte[TR_EXTENSION]);
				printf ("\", and length already exists on TRDOS disk image.\n");
				
				if (verbose)
				{
					printf ("You can use option --force and add this binary file on TRDOS image anyway.\n");
				}
				fatal_error_found = 1;
			}
			else
			{
				if (verbose) printf ("File with same name is already on TRDOS image, but ignoring it, because option --force was used.\n");
			}
		}
		
		/* if any check failed, don't continue */
		if (fatal_error_found)
		{
			fclose (sourcefile);
			fclose (targetfile);
			exit (EXIT_FAILURE);
		}
		
		if (DEBUG) printf ("DEBUG Checks done, writing binary file... \n");
		
		/* print filename with proper conversion of semigraphics and BASIC tokens */
		if (verbose)
		{
			for (i = 0; i < 8; i++) printzxchars(trdos_filename.byte[i]);
			printf (".");
			printzxchars(trdos_filename.byte[8]);
			printf ("\n");
		}
		
		/*
			DATA SOUBORU
			najít volnou položku adresáře
			vytvořit záznam v adresáři
			dle délky upravit záznam v systémovém sektoru
			seek na první volný sektor a stopu v cílovém souboru
			přenést data z binárního souboru do sektorů
			poslední sektor, pokud nekompletní, doplnit nulami
		*/
		sector_number = track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR];
		track_number = track.sector[INFO_SEC].byte[OFFSET_FIRSTTRACK];
		free_sectors = track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1]*256+track.sector[INFO_SEC].byte[OFFSET_FREESECTORS];
		fseek (sourcefile, 0, SEEK_SET);
		fseek (targetfile, sector_number*SEC_SIZE+track_number*TRK_SIZE*SEC_SIZE, SEEK_SET);
		
		if (DEBUG)
		{
			printf ("DEBUG first sector %hhu\n", sector_number);
			printf ("DEBUG first track %hhu\n", track_number);
			printf ("DEBUG free sectors %hu\n", free_sectors);
		}
		
		/* transfer of data from binary file in to file in TRDOS image */
		file_length = 0;	/* byte counter */
		sector_length = 0;	/* sector counter */
		
		while (fread (&byte, sizeof (uint8_t), 1, sourcefile) == 1)
		{
			/* at the beginning of every sector */
			
			if ((file_length % 256) == 0)
			{
				if (verbose) printf ("[%u,%u] ", track_number, sector_number);
				
				/* move "pointer" to the next first free sector/track */
				sector_length++;	/* length of file in sectors */
				sector_number++;	/* pointer to the disk structure */
				if (sector_number > 15)
				{
					sector_number = 0;
					track_number++;
				}
				
				free_sectors--;
				if (free_sectors < 0)
				{
					/* should not happen, free space was verified before */
					colored_error ();
					printf ("Out of disk space.\n");
					fclose (sourcefile);
					fclose (targetfile);
					exit (EXIT_FAILURE);
				}
			}

			if (!dryrun)
			{
				/* write byte */
				if (fwrite (&byte, sizeof (uint8_t), 1, targetfile) != 1)
				{
					printf ("Writing in TRDOS image file %s failed: %s\n", target_filename, strerror(errno));
					fclose (sourcefile);
					fclose (targetfile);
					exit (EXIT_FAILURE);
				}
			}
			
			/* byte written */
			file_length++;
		}
		
		/* check if last sector is written whole or partially - if partially, write zeroes in unused space */
		if (verbose) printf ("\n");
		
		if ((file_length % 256) != 0)
		{
			if (verbose) printf ("Aligning last sector with %hhu zeroes.", file_length % 256);
			
			if (!dryrun)
			{
				byte = 0;
				/* fill unfinished sector with zeroes */
				for (i = file_length % 256; i < 255; i++)
				{
					if (fwrite (&byte, sizeof (uint8_t), 1, targetfile) != 1)
					{
						printf ("Writing in TRDOS image file %s failed: %s\n", target_filename, strerror(errno));
						fclose (sourcefile);
						fclose (targetfile);
						exit (EXIT_FAILURE);
					}
				}
			}
		}
		else
			if (verbose) printf ("Last sector was filled with data from binary file.\n");
			
		/* create record in TRDOS directory, update system sector -> determine position in directory  */
		dir_sector = track.sector[INFO_SEC].byte[OFFSET_FILES] / 16;	/* in which sector write filename of new file */
		directory_offset = (track.sector[INFO_SEC].byte[OFFSET_FILES] % 16)*16;	/* which item in directory will be modified */
		

		
		/* filename + extension */
		for (i = 0; i < 9; i++)
		{
			track.sector[dir_sector].byte[directory_offset+i] = trdos_filename.byte[i];
		}
		/* start address of CODE */
		track.sector[dir_sector].byte[directory_offset+9] = start_address % 256;
		track.sector[dir_sector].byte[directory_offset+10] = start_address / 256;
		/* length of CODE */
		track.sector[dir_sector].byte[directory_offset+11] = file_length % 256;
		track.sector[dir_sector].byte[directory_offset+12] = file_length / 256;
		/* length in sectors */
		track.sector[dir_sector].byte[directory_offset+13] = sector_length;
		/* file position */
		track.sector[dir_sector].byte[directory_offset+14] = track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR];
		track.sector[dir_sector].byte[directory_offset+15] = track.sector[INFO_SEC].byte[OFFSET_FIRSTTRACK];
		
		/* modificaion of parameters in TRDOS system sector 9 */
		if (verbose) printf ("\nNew first track and sector %u, %u\n", track_number, sector_number);
		track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR] = sector_number;
		track.sector[INFO_SEC].byte[OFFSET_FIRSTTRACK] = track_number;
		track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] = free_sectors % 256;
		track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] = free_sectors / 256;
		track.sector[INFO_SEC].byte[OFFSET_FILES]++;
		
		if (!dryrun)
		{
			/* write whole directory with all changes */
			fseek (targetfile, 0, SEEK_SET);
			
			if (fwrite (&track, sizeof (trdos_track), 1, targetfile) != 1)
			{
				printf ("Writing in TRDOS image file %s failed: %s\n", target_filename, strerror(errno));
				fclose (sourcefile);
				fclose (targetfile);
				exit (EXIT_FAILURE);
			}
		}
		/* done */
		if (verbose) printf ("Free: %hu\nO.K.\n", free_sectors);
	}
	else
	{
		colored_error ();
		printf ("Reading of directory from TRDOS disk image %s failed.\n", target_filename);
		fclose (sourcefile);
		fclose (targetfile);
		exit (EXIT_FAILURE);
	}
}


uint16_t string_to_address (char* s)
{
	/* will convert parametr in to 16 bit start address */
	uint16_t address;
	
/*	if (atoi (s) < 0) address == 0;
		else
			if (atoi (s) > 65535) address == 65535;
			else
				address == atoi (s);*/

	address = strtol (s, NULL, 10);
	
	return (address);
}


void version ()
{
	printf ("version %s \n", VERSION);
}


void help ()
{
	printf ("BIN2TRD is a utility for adding binary files as CODE in to TRDOS disk image.\n");
	printf ("Usage: bin2trd [OPTIONS] [SOURCE] [TARGET]\n");
	printf ("[SOURCE] must be any binary file from 1 to 65535 bytes long.\n");
	printf ("[TARGET] must be TRDOS disk image\n");
	printf ("  -V, --verbose  increase verbosity level\n");
	printf ("  -a, --address  start address of target file\n");
	printf ("  -f, --filename name of target file\n");
	printf ("  -C, --color    enable color output, it will highlight some values\n");
	printf ("  --dry-run      will simulate whole procedure except writing in TRD image\n");
	printf ("  -h, --help     this text\n");
	printf ("  -v, --version  version\n");
	printf ("\n");
	printf ("If you want to list content of TRD image, utility lstrd can help.\n");
	printf ("\n");
	printf ("My website https://cygnus.speccy.cz\n");
// 	printf ("Project website https://sourceforge.net/projects/zxspectrumutils\n");
}


int main(int argc, char *argv[])
{
	char *source_filename = NULL;
	char *target_filename = NULL;
	Ttrdosfilename trdos_filename;
	uint16_t start_address;
	
	unsigned int counter;
	int switches;
	unsigned char correct;
	int returnvalue;
	uint32_t crc32;
	
	unsigned int i;

	enum {option, trdosfilename, startaddress} parsermode;

	/* input file is parameter before last, output file must be last parameter */
	if (DEBUG) printf ("DEBUG: argc - %i\n", argc);
	
	verbose = 0;
	colors = 0;

	start_address = 0;		/* start address */

	for (i = 0; i < 8; i++)		/* filename on TRDOS disk - empty 8 spaces plus extension, always C = CODE */
		trdos_filename.byte[i] = ' ';
	trdos_filename.byte[8] = 'C';
	
	parsermode = option;

	switches = 1;
	for (counter = 1; counter < argc; counter++)
	{
		if (DEBUG) printf ("++ %i ++ %s \n",counter, argv[counter]);
		correct = 0;

		switch (parsermode)
		{
			case option:
				if (!strcmp(argv[counter], "-h") || (!strcmp(argv[counter], "--help")))
				{
					help ();
					switches++;
					correct = 1;
					return (EXIT_SUCCESS);
				}
				if (!strcmp(argv[counter], "-v") || (!strcmp(argv[counter], "--version")))
				{
					version ();
					switches++;
					correct = 1;
					return (EXIT_SUCCESS);
				}
				if (!strcmp(argv[counter], "-V") || (!strcmp(argv[counter], "--verbose")))
				{
					verbose = 1;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "-a") || (!strcmp(argv[counter], "--address")))
				{
					parsermode = startaddress;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "-f") || (!strcmp(argv[counter], "--filename")))
				{
					parsermode = trdosfilename;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "--dry-run"))
				{
					/* simulate everything, but don't write binary file in target trdos image file */
					dryrun = 1;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "-C") || (!strcmp(argv[counter], "--color")) || (!strcmp(argv[counter], "--colors")))
				{
					/* colors enabled */
					colors = 1;
					switches++;
					correct = 1;
				}
				if ((correct == 0) && !((counter == (argc-1)) || (counter==(argc-2))))
				{
					/* parameter before last can be filename of source file */
					/* last parameter is filename of target TRD image */
					colored_error ();
					printf ("Unknown parameter - \"%s\"\n", argv[counter]);
					help ();
					exit (EXIT_FAILURE);
				}
				break;
			case trdosfilename:
				if (DEBUG) printf ("+++ strlen(argv[counter]) %lu \n", strlen(argv[counter]));
				if (strlen(argv[counter]) > 8)
				{
					colored_warning ();
					printf ("TRDOS filename is longer than 8 characters.\n");
				}
				for (i = 0; i < 8; i++)
				{
					if (strlen(argv[counter]) > i)
						trdos_filename.byte[i] = argv[counter][i];
					else
						trdos_filename.byte[i] = ' ';
				}
				switches++;
				parsermode = option;
				break;
			case startaddress:
				if (argv[counter][0] == '-')
				{
					/* numeric parameter must be 0..65535 */
					switches++;
				}
				else
				{
					start_address = string_to_address (argv[counter]);
					switches++;
					correct = 1;
				}
				if (correct == 0)
				{
					colored_error ();
					printf ("Unknown parameter - \"%s\"\n", argv[counter]);
				}
				parsermode = option;
				break;
		}
	}
	if (DEBUG)
		printf ("switches %hhu argc %hhu\n", switches, argc);
	
	if (switches != argc-2)
	{
		colored_error ();
		printf ("At least one of filenames is missing.\n");
		exit (EXIT_FAILURE);
	}
	else
	{
		source_filename = argv[argc-2];
		target_filename = argv[argc-1];
		
		if (verbose)
		{
			printf ("[SOURCE] %s -> [TARGET] %s as [TRDOS FILE] \"", source_filename, target_filename );
			for (i = 0; i < 8; i++) printzxchars(trdos_filename.byte[i]);
			printf (".");
			printzxchars(trdos_filename.byte[8]);
			printf ("\" from address %hu \n", start_address);
		}

		return (bintotrd (source_filename, target_filename, trdos_filename, start_address));
	}
}
