/*
* This file is part of CREATETRD.
*
* Copyright (C) 2003 Pavel Cejka <pavel.cejka at kapsa.club.cz>
*
* Modifications (C) 2015 ub880d <ub880d@users.sf.net> (mainly for pedantic compile)
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or (at
* your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>		// strcpy
// #include <stdint.h>		// integer types
#include "trdos_structure.h"

#define VERSION		"0.6"
#define	DEBUG		0

/* global options */
int	verbose = 0;
int	force = 0;
int	colors = 0;

/* selection of ANSI colors */
#define CRED     "\x1b[31m"
#define CGREEN   "\x1b[32m"
#define CYELLOW  "\x1b[33m"
#define CBLUE    "\x1b[34m"
#define CMAGEN   "\x1b[35m"
#define CCYAN    "\x1b[36m"
#define CBRED    "\x1b[91m"
#define CBGREEN  "\x1b[92m"
#define CBYELLOW "\x1b[93m"
#define CBWHITE  "\x1b[97m"
#define CRESET   "\x1b[0m"

typedef enum {none, black, red, green, yellow, white, brightred, brightgreen, brightyellow, brightwhite} listofcolors_t;

/* separated in function, because this will be different for different operating systems */
void colorize (listofcolors_t c)
{
	/* only if colors are globally enabled */
	if (colors)
	{
		/* TODO: different colors for terminals with white background */
		switch (c)
		{
			case red:
				printf ("%s", CRED);
				break;
			case green:
				printf ("%s", CGREEN);
				break;
			case brightgreen:
				printf ("%s", CBGREEN);
				break;
			case brightyellow:
				printf ("%s", CBYELLOW);
				break;
			case brightwhite:
				printf ("%s", CBWHITE);
				break;
			default:
				printf ("%s", CRESET);
				break;
		}
	}
}

void colored_warning (void)
{
	colorize (brightyellow);
	printf ("Warning: ");
	colorize (none);
}

void colored_error (void)
{
	colorize (red);
	printf ("Error! ");
	colorize (none);
}

// return 1 if the file specified by the filename exists
int is_file_exists(const char *filename)
{
    int exists = 0;
    FILE *fp = fopen(filename, "r");
    
    if (fp != NULL)
    {
        exists = 1;
        fclose(fp);
    }

    return exists;
}

void create (char *filename, char *diskname, int diskformatcode)
{
	int i;
	trdos_sector empty_sector;
	trdos_track track;
	int number_of_tracks;
	FILE *f;

	/* inicializace tracku */
	for (i = 0; i < SEC_SIZE; i++) empty_sector.byte[i] = 0;
	for (i = 0; i < 16; i++) track.sector[i] = empty_sector;

	/* tvorba 8. sektoru číslo 7 */
/*	track.sector[INFO_SEC].byte[OFFSET_FIRSTSECTOR] = 0;		// není potřeba zapisovat*/
	track.sector[INFO_SEC].byte[OFFSET_FIRSTTRACK] = 1;
	track.sector[INFO_SEC].byte[OFFSET_DISKFORMAT] = diskformatcode;
	switch (diskformatcode)
	{
		case 22: /* 80DS */
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] = 240;
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] = 9;
			number_of_tracks = 80*2-1;
			break;
		case 23: /* 40DS */
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] = 240;
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] = 4;
			number_of_tracks = 40*2-1;
			break;
		case 24: /* 80SS */
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] = 240;
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] = 4;
			number_of_tracks = 80-1;
			break;
		case 25: /* 40SS */
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS] = 112;
			track.sector[INFO_SEC].byte[OFFSET_FREESECTORS+1] = 2;
			number_of_tracks = 40-1;
			break;
		default:
			/* this should never happen */
			colored_error ();
			printf ("internal error, bad type od disk image format %i\n", diskformatcode);
			exit (EXIT_FAILURE);
			break;
	}
/*	track.sector[INFO_SEC].byte[OFFSET_FILES] = 0;			// není potřeba zapisovat */
	track.sector[INFO_SEC].byte[OFFSET_TRDOSIDENT] = 16;
/*	track.sector[INFO_SEC].byte[OFFSET_PASSWORD] = 0;		// není potřeba zapisovat */
/*	track.sector[INFO_SEC].byte[OFFSET_DELETEDFILES] = 0;		// není potřeba zapisovat */
	i = 0;
	while ((diskname != NULL) && (diskname[i] != 0) && (i < 10))
	{
		track.sector[INFO_SEC].byte[OFFSET_DISKNAME+i] = diskname[i];
		i++;
	}
	
	/* vytvořit soubor a ohlásit chybu, když se to nepovede */
	f = fopen (filename, "w");
	if (f == NULL) 
	{
		colored_error ();
		printf ("couldn't open output file for writing.\n");
		exit (EXIT_FAILURE);
	}
	/* zapsat první track */
	fwrite (&track, sizeof(trdos_track), 1, f);
	/* vynulovat systémový sektor (už nebude potřeba a proměnnou pro track použiju jinak) a zapsat tracky pro data */
	track.sector[INFO_SEC] = empty_sector;
	for (i = 0; i < number_of_tracks; i++) fwrite (&track, sizeof(trdos_track), 1, f);
	/* zavřít soubor */
	fclose (f);
}

void version ()
{
	printf ("%s \n", VERSION);
}

void help ()
{
	printf ("CREATETRD is simple utility for creating TRDOS disk empty image files.\n");
	printf ("Usage:   createtrd [OPTIONS] [CREATED FILENAME]\n");
	printf ("Example: createtrd -n MUJPOKUS -f 80DS mujpokus.trd\n");
	printf ("  -V, --verbose    increase verbosity level\n");
	printf ("  -f, --format     disk format (40SS, 40DS, 80SS, 80DS)\n");
	printf ("  --force          rewrite existing file without asking\n");
	printf ("  -n, --diskname   disk name long max. 10 characters (default name is empty)\n");
	printf ("  -h, --help       this text\n");
	printf ("  -v, --version    version\n");
	printf ("\n");
	printf ("If you want to list content of TRD image, utility lstrd can help.\n");
	printf ("\n");
	printf ("My website https://cygnus.speccy.cz\n");
// 	printf ("Project website https://sourceforge.net/projects/zxspectrumutils\n");
}

int main(int argc, char *argv[])
{
	char *filename;
	char *diskname;
	int counter;
	int switches;
	int correct;
	int diskformatcode;
	
	enum {option, trdosformat, trdosdiskname} parsermode;
	
	/* disk image filename is last parameter, other parameters must be known options */
	if (DEBUG) printf ("DEBUG: argc - %i\n", argc);

	verbose = 0;
	colors = 0;

	switches = 1;		/* number of switches */
	parsermode = option;

	diskformatcode = 22;	/* defaul 80DS = 80 tracks, double sided disk = 640kB / 2455 free sectors*/
	diskname = NULL;	/* until one option will determine TRDOS disk name */

	for (counter = 1; counter < argc; counter++)
	{
		if (DEBUG) printf ("++DEBUG++ %i ++ %s \n",counter, argv[counter]);
		correct=0;

		switch (parsermode)
		{
			case option:
				if (!strcmp(argv[counter],"-h") || (!strcmp(argv[counter],"--help")))
				{
	    				help ();
					switches++;
					correct = 1;
					return (EXIT_SUCCESS);
				}
				if (!strcmp(argv[counter],"-v") || (!strcmp(argv[counter],"--version")))
				{
					version ();
					switches++;
					correct = 1;
					return (EXIT_SUCCESS);
				}
				if (!strcmp(argv[counter], "-V") || (!strcmp(argv[counter], "--verbose")))
				{
					verbose = 1;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter],"-f") || (!strcmp(argv[counter],"--format")))
				{
					parsermode = 1;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter],"-n") || (!strcmp(argv[counter],"--diskname")))
				{
					parsermode = 2;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "--force"))
				{
					/* simulate everything, but don't write binary file in target trdos image file */
					force = 1;
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter], "-C") || (!strcmp(argv[counter], "--color")) || (!strcmp(argv[counter], "--colors")))
				{
					/* colors enabled */
					colors = 1;
					switches++;
					correct = 1;
				}
				if ((correct == 0) && (counter != (argc-1)))
				{
					/* last parameter should be disk image filename */
					colored_error ();
					printf ("unknown parametr - \"%s\"\n", argv[counter]);
					help ();
					return (EXIT_FAILURE);
				}
			break;
			case trdosdiskname:
				if ((argv[counter][0]) != '-')
				{
					parsermode = 0;
					switches++;
					// if (verbose) printf ("format 40 tracks single sided (624 free sectors)\n");
					correct = 1;
					diskname = argv [counter];
					
					
					if (strlen(diskname) > 10)
					{
						colored_warning ();
						printf ("TRDOS diskname is too long, should be up to 8 or 10 characters.\n");
					}
					if (verbose) printf ("TRDOS diskname %s\n", diskname);
				}
				if (correct == 0)
				{
					colored_error ();
					printf ("Missing TRDOS diskname, option %s detected instead.\n", argv[counter]);
					help ();
					return (EXIT_FAILURE);
				}
			break;
			case trdosformat:
				if (!strcmp(argv[counter],"80DS"))
				{
					parsermode = 0;
					diskformatcode = 22;
					if (verbose) printf ("format 80 tracks double sided (2544 free sectors)\n");
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter],"40DS"))
				{
					parsermode = 0;
					diskformatcode = 23;
					if (verbose) printf ("format 40 tracks double sided (1264 free sectors)\n");
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter],"80SS"))
				{
					parsermode = 0;
					diskformatcode = 24;
					if (verbose) printf ("format 80 tracks single sided (1264 free sectors)\n");
					switches++;
					correct = 1;
				}
				if (!strcmp(argv[counter],"40SS"))
				{
					parsermode = 0;
					diskformatcode = 25;
					if (verbose) printf ("format 40 tracks single sided (624 free sectors)\n");
					switches++;
					correct = 1;
				}
				if (correct == 0)
				{
					colored_error ();
					printf ("Unknown format argument \"%s\". Please use 40SS, 80SS, 40DS or 80DD.\n", argv[counter]);
					help ();
					exit (EXIT_FAILURE);
				}
			break;
			default:
				colored_error ();
				printf ("unknown parsermode - %d\n", parsermode);
			break;
		}
	}

	// last parametr
	filename = argv[argc-1];

	if (verbose) printf ("Disk image filename: %s\n", filename);

	if (switches != argc-1)
	{
		colored_error ();
		printf ("Filename of TRDOS disk image is missing.\n");
		exit (EXIT_FAILURE);
	}
	else
	{
		/* don't rewrite disk image if already exists and option --force is not used */
		int file_exists;
		file_exists = is_file_exists(filename);
		
		if ((file_exists == 0) || (file_exists == 1 && force == 1))
		{
			if (verbose && file_exists == 1 && force == 1)
			{
				colored_warning ();
				printf ("I am rewriting existing TRDOS disk image because parameter --force.\n");
			}
			create (filename, diskname, diskformatcode);
			return (EXIT_SUCCESS);
		}
		else
		{
			colored_error ();
			printf ("TRDOS disk image is already exists.\n");
			exit (EXIT_FAILURE);
		}
	}
}
